/** @file   mine.cpp
 * @brief   Implementation of Mine - class.
 * @version $Revision: 1.3 $
 * @author  Tomi Lamminsaari
 */

#include "mine.h"
#include "gameanims.h"
#include "animplayer.h"
#include "soundsamples.h"
#include "settings.h"
#include "www_map.h"
#include "warglobals.h"
#include "sensorcontroller.h"
#include "redrawqueue.h"
#include "AnimId.h"

using namespace eng2d;

namespace WeWantWar {

/** Constructor
 */
Mine::Mine( MineType minetype ) :
  GameObject(),
  m_minetype( minetype )
{
  switch ( minetype ) {
    case ( ANTI_INFANTRY ): {
      this->initInfantryMine();
      break;
    }
  }
}



/** Destructor
 */
Mine::~Mine()
{
}


/** Updates this mine.
 */
void Mine::update()
{
  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }
  if ( this->state() == GameObject::STATE_DYING ) {
    if ( this->getCounter( DETONATING_COUNTER_INDEX ) < 0 ) {
      this->kill();
    }
    return;
  }
  BaseController* pC = this->getController();
  pC->update();
  if ( pC->shoot() == 1 ) {
    this->state( GameObject::STATE_DYING );
    this->setCounter( DETONATING_COUNTER_INDEX, m_detonatingDelay );
    this->makeSound( GameObject::SND_ATTACK );
  }
}



/** Redraws this mine
 */
void Mine::redraw( RedrawQueue* pQueue )
{
  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }
  
  int mx = MobileObject::m_position.intX() - Map::scrollX;
  int my = MobileObject::m_position.intY() - Map::scrollY;

  RLE_SPRITE* pS = AnimatedObject::m_animation.currentFrame().asRleSprite();
  mx -= pS->w / 2;
  my -= pS->h / 2;
  
  pQueue->add( RedrawQueue::PRI_BELOW_NORMAL, mx,my,
               RedrawQueue::SPTYPE_RLE, pS );
}



/** Makes sound
 */
void Mine::makeSound( GameObject::SoundID id ) const
{
  switch ( id ) {
    case ( GameObject::SND_DIE ): {
      Sound::playSample( SMP_GRENADE, false );
      break;
    }
    case ( GameObject::SND_ATTACK ): {
      // Detonating sound
      Sound::playSample( SMP_MINE, false );
      break;
    }
  }
}



/** We got hit by a bullet.
 */
bool Mine::hitByBullet( Bullet* pB )
{
  return false;
}



/** Kills this object
 */
void Mine::kill()
{
  if ( this->state() == GameObject::STATE_LIVING ) {
    return;
  }
  
  this->detonateMine();
  this->makeSound( GameObject::SND_DIE );
  this->state( GameObject::STATE_KILLED );
}




///
/// Getter methods
/// ==============

/** Type of this GameObject
 */
ObjectID::Type Mine::objectType() const
{
  return ObjectID::TYPE_MINE;
}



/** Are we in reloading - state
 */
bool Mine::reloading() const
{
  return false;
}



/** Minetype
 */
Mine::MineType Mine::mineType() const
{
  return m_minetype;
}




///
/// Private or Protected methods
/// ============================

/** Initializes the infantry mine
 */
void Mine::initInfantryMine()
{
  int oid = ObjectID::TYPE_MINE;
  m_fragmentDamage = Settings::intObjProp( oid, "infantry_damage:" );
  m_fragmentVelocity = Settings::floatObjProp( oid, "infantry_velocity:" );
  m_fragmentCount = Settings::intObjProp( oid, "infantry_fragments:" );
  m_detonatingDelay = Settings::intObjProp( oid, "infantry_detonate:" );
  m_fragmentRange = Settings::floatObjProp( oid, "infantry_range:" );

  const Animation& anim = GameAnims::findAnimation( AnimId::KMines );
  this->setAnimation( anim, GameAnims::EIdle );
  this->boundingSphere( Settings::floatObjProp( oid, "infantry_bsphere:" ) );
  this->setArmor( 100000 );
  this->setProperties( 0 );
  
  SensorController* pC = new SensorController( this );
  pC->m_senseRange = Settings::floatObjProp( oid, "infantry_senserange:" );
  this->setController( pC );
}



/** Detonates this mine
 */
void Mine::detonateMine()
{
  switch ( m_minetype ) {
    case ( ANTI_INFANTRY ): {
      Vec2D p( this->position() );
      
      float angleStep = 256.0 / static_cast<float>( m_fragmentCount );
      for ( float a=0; a < 256; a += angleStep ) {
        Vec2D dirVec( 0, -m_fragmentVelocity );
        dirVec.rotate( a );
        
        Bullet* pB = BulletTable::createBullet( this, m_position,
                                                Bullet::EInfantryMines );

        pB->setVelocity( dirVec );
        pB->iRange = m_fragmentRange;
        pB->iDamage = m_fragmentDamage;
        
        WarGlobals::pBulletManager->spawnBullet( pB );
      }
      const Animation& explosionAnim = GameAnims::findAnimation( AnimId::KExplosionGrenade );
      AnimPlayer::spawn( explosionAnim, this->position(), 0 );
      break;
    }
  }
}


} // end of namespace
